/* $Id: order-active.c,v 1.2 1999/01/30 01:06:41 ericb Exp $ */
/* Copyright (C) 1998 - 1999, Hewlett-Packard Company, all rights reserved. */
/* Written by Eric Backus */

/* Test that inactive channels don't mess up resample and order data.

   This test requires a tach signal going into the tach1 input, of
   between 250 and 500 Hz, with TTL signal levels. */

   

#include <math.h>
#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <unistd.h>
#include "e1432.h"

#define	NMOD_MAX	4
#define	NCHAN_MAX	(NMOD_MAX * E1432_INPUT_CHANS)
#define	BLOCKSIZE	1024
#define	SPAN		5000
#define	NSCAN		10

/* Wrap this around all the many function calls which might fail */
#ifdef	__lint
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (func)
#else
#define	CHECK(func)	\
do {\
    int _s = (func);\
    if (_s < 0)\
    {\
	(void) fprintf(stderr, "%s: %s: returned %d\n", progname, #func, _s);\
	return _s;\
    }\
} while (0)
#endif

static const volatile char rcsid[] =
"@(#)$Id: order-active.c,v 1.2 1999/01/30 01:06:41 ericb Exp $";
static const char *progname;

static int
init(int nmod, SHORTSIZ16 *laddr, E1432ID *hw, int *group,
     int *nchan, SHORTSIZ16 *chan_list)
{
    struct e1432_hwconfig hwconfig[NMOD_MAX];
    int     i, nc, tachs;

    /* Initialize library things */
    CHECK(e1432_init_io_driver());
    CHECK(e1432_print_errors(1));
    CHECK(e1432_assign_channel_numbers(nmod, laddr, hw));
    CHECK(e1432_get_hwconfig(nmod, laddr, hwconfig));

    /* How many channels should we use? */
    nc = 0;
    for (i = 0; i < nmod; i++)
	nc += hwconfig[i].input_chans;
    if (nc > NCHAN_MAX)
	nc = NCHAN_MAX;
    /* Silently ignore attempts to use fewer than 3 channels, so we
       don't end up with no channels active. */
    if (*nchan < 3 && *nchan != -1)
	*nchan = 3;
    if (nc < 3)
    {
	(void) fprintf(stderr, "%s: need at least three input channels\n",
		       progname);
	return -1;
    }
    if (nc > *nchan && *nchan != -1)
	nc = *nchan;
    *nchan = nc;

    for (i = 0; i < nc; i++)
	chan_list[i] = E1432_INPUT_CHAN(i + 1);

    *group = e1432_create_channel_group(*hw, nc, chan_list);
    if (*group >= 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_create_channel_group: returned %d\n",
		       progname, *group);
	return -1;
    }

    tachs = 0;
    for (i = 0; i < nmod; i++)
	tachs += hwconfig[i].tach_chans;

    if (tachs == 0)
    {
	(void) fprintf(stderr, "%s: requires a tach channel\n", progname);
	return -1;
    }

    return 0;
}

/*ARGSUSED*/
static int
setup(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list)
{
    CHECK(e1432_set_append_status(hw, group, E1432_APPEND_STATUS_ON));
    CHECK(e1432_set_blocksize(hw, group, BLOCKSIZE));
    CHECK(e1432_set_span(hw, group, SPAN));
    CHECK(e1432_set_data_size(hw, group, E1432_DATA_SIZE_32_SERV));

    /* Minimal setup for order tracking */
    CHECK(e1432_set_calc_data(hw, group, E1432_DATA_ORDER));
    CHECK(e1432_set_decimation_output(hw, group, E1432_MULTIPASS));

    CHECK(e1432_set_active(hw, E1432_TACH_CHAN(1), E1432_CHANNEL_ON));
    CHECK(e1432_set_arm_channel(hw, E1432_TACH_CHAN(1), E1432_CHANNEL_ON));

    CHECK(e1432_set_trigger_level(hw, E1432_TACH_CHAN(1),
				  E1432_TRIGGER_LEVEL_LOWER, 0.8));
    CHECK(e1432_set_trigger_level(hw, E1432_TACH_CHAN(1),
				  E1432_TRIGGER_LEVEL_UPPER, 1.4));

    CHECK(e1432_set_delta_order(hw, E1432_TACH_CHAN(1), 1.0));
    CHECK(e1432_set_max_order(hw, E1432_TACH_CHAN(1), 10.0));
    CHECK(e1432_set_rpm_high(hw, E1432_TACH_CHAN(1), SPAN * 60 / 10));
    CHECK(e1432_set_rpm_low(hw, E1432_TACH_CHAN(1), SPAN * 60 / 10 / 16));

    return 0;
}

static int
set_active1(E1432ID hw, int nchan, SHORTSIZ16 *chan_list, int *active)
{
    int     i;

    for (i = 0; i < nchan; i++)
    {
	CHECK(e1432_set_active(hw, chan_list[i], E1432_CHANNEL_ON));
	active[i] = 1;
    }

    /* Turn off first channel */
    CHECK(e1432_set_active(hw, chan_list[0], E1432_CHANNEL_OFF));
    active[0] = 0;

    /* Turn off middle channel */
    CHECK(e1432_set_active(hw, chan_list[nchan / 2], E1432_CHANNEL_OFF));
    active[nchan / 2] = 0;

    return 0;
}

static int
set_active2(E1432ID hw, int nchan, SHORTSIZ16 *chan_list, int *active)
{
    int     i;

    for (i = 0; i < nchan; i++)
    {
	CHECK(e1432_set_active(hw, chan_list[i], E1432_CHANNEL_OFF));
	active[i] = 0;
    }

    /* Turn on second channel */
    if (nchan > 0)
    {
	CHECK(e1432_set_active(hw, chan_list[1], E1432_CHANNEL_ON));
	active[1] = 1;
    }

    /* Turn on middle channel */
    if (nchan > 1)
    {
	CHECK(e1432_set_active(hw, chan_list[nchan / 2 - 1],
			       E1432_CHANNEL_ON));
	active[nchan / 2 - 1] = 1;
    }

    return 0;
}

static int
set_active3(E1432ID hw, int nchan, SHORTSIZ16 *chan_list, int *active)
{
    int     i;

    for (i = 0; i < nchan; i++)
    {
	CHECK(e1432_set_active(hw, chan_list[i], E1432_CHANNEL_OFF));
	active[i] = 0;
    }

    /* Turn on third through eighth channels */
    for (i = 2; i < 8; i++)
    {
	if (i >= nchan)
	    break;

	CHECK(e1432_set_active(hw, chan_list[i], E1432_CHANNEL_ON));
	active[i] = 1;
    }

    return 0;
}

static int
check_trailer(struct e1432_trailer *trailer, FLOATSIZ32 clock_freq,
	      double span, int chan, int type, double rpm1)
{
    double  tmp;
    int     df2, df5, tdf2;

    if (trailer->zoom_corr != 0)
    {
	/* Zoom correction is not currently implemented */
	(void) fprintf(stderr,
		       "%s: trailer zoom corr non-zero: %g (0x%lx)\n",
		       progname, trailer->zoom_corr,
		       *(long *) &trailer->zoom_corr);
	return -1;
    }
    if (trailer->gap < 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer gap negative: 0x%lx\n",
		       progname, trailer->gap);
	return -1;
    }
    if (rpm1 >= 0)
    {
	tmp = trailer->rpm1 - rpm1;
	if (tmp < 0)
	    tmp = -tmp;
	/* Empirically determined - the worst I observed was 21 RPM */
	if (tmp > 30)
	{
	    (void) fprintf(stderr,
			   "%s: trailer rpm1 incorrect: %g (0x%lx) expected %g\n",
			   progname, trailer->rpm1,
			   *(long *) &trailer->rpm1,
			   rpm1);
	    return -1;
	}
    }
    if (trailer->rpm2 != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rpm2 non-zero: %g (0x%lx)\n",
		       progname, trailer->rpm2,
		       *(long *) &trailer->rpm2);
	return -1;
    }
    if (trailer->peak != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer peak non-zero: %g (0x%lx)\n",
		       progname, trailer->peak,
		       *(long *) &trailer->peak);
	return -1;
    }
    if (trailer->rms != 0)
    {
	(void) fprintf(stderr,
		       "%s: trailer rms non-zero: %g (0x%lx)\n",
		       progname, trailer->rms,
		       *(long *) &trailer->rms);
	return -1;
    }

    /* Compute df2 and df5 from clock_freq and span */
    tmp = span * 2.56;
    df2 = 0;
    df5 = 0;
    while (tmp < clock_freq * 0.9999)
    {
	df2++;
	tmp *= 2;
    }
    if (tmp > clock_freq * 1.0001)
    {
	tmp /= 8;
	tmp *= 5;
	df2 -= 3;
	df5++;
	if (tmp > clock_freq * 1.0001 || tmp < clock_freq * 0.9999)
	{
	    (void) fprintf(stderr,
			   "%s: invalid span/clock_freq combination: %g/%g\n",
			   progname, span, clock_freq);
	    return -1;
	}
    }

    /* For resample and order data, allow trailer df2 to be off by one
       or two, to allow for a sweeping input signal. */
    tdf2 = (trailer->info & E1432_TRAILER_INFO_DEC_2_MASK) >>
	E1432_TRAILER_INFO_DEC_2_SHIFT;
    if ((type == 0 && df2 != tdf2)
	|| tdf2 < df2 || tdf2 > df2 + 2)
    {
	(void) fprintf(stderr,
		       "%s: trailer info df2 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df2);
	return -1;
    }
    if (df5 != ((trailer->info & E1432_TRAILER_INFO_DEC_5) != 0))
    {
	(void) fprintf(stderr,
		       "%s: trailer info df5 mismatch: 0x%8.8lx, %d\n",
		       progname, trailer->info, df5);
	return -1;
    }

    if (((trailer->info & E1432_TRAILER_INFO_CHAN_MASK) >>
	 E1432_TRAILER_INFO_CHAN_SHIFT) != chan - 1)
    {
	(void) fprintf(stderr,
		       "%s: trailer info chan mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, chan - 1);
	return -1;
    }
    if (((trailer->info & E1432_TRAILER_INFO_TYPE_MASK) >>
	 E1432_TRAILER_INFO_TYPE_SHIFT) != type)
    {
	(void) fprintf(stderr,
		       "%s: trailer info type mismatch: 0x%8.8lx, 0x%x\n",
		       progname, trailer->info, type);
	return -1;
    }

    return 0;
}

static int
wait_block_avail(E1432ID hw, int group, int scan, int verbose,
		 long blocksize, double span)
{
    clock_t start, timeout;
    int     status;

    /* Allow extra overhead, because resample calculations are slow */
    timeout = (4 + 2 * (blocksize / (span * 2.56))) * CLOCKS_PER_SEC;
    if (verbose > 2)
	(void) printf("Waiting %g sec for block available\n",
		      (double) timeout / CLOCKS_PER_SEC);
    start = clock();
    while ((status = e1432_block_available(hw, group)) == 0)
	if (clock() - start > timeout &&
	    (status = e1432_block_available(hw, group)) == 0)
	{
	    (void) fprintf(stderr, "%s: e1432_block_available: "
			   "timeout waiting %g sec\n",
			   progname, (double) timeout / CLOCKS_PER_SEC);
	    return -1;
	}
    if (status < 0)
    {
	(void) fprintf(stderr,
		       "%s: e1432_block_available: returned %d\n",
		       progname, status);
	return -1;
    }
    if (verbose > 0)
	(void) printf("Scan %d block available found\n", scan);

    return 0;
}

static int
run(E1432ID hw, int group, int nchan, SHORTSIZ16 *chan_list,
    double *signal_list, int *active, double freq, int verbose)
{
    FLOATSIZ64 buf[BLOCKSIZE];
    LONGSIZ32 count;
    struct e1432_trailer trailer;
    double  rms;
    long    i;
    int     scan, chan, type, data_type;

    CHECK(e1432_init_measure(hw, group));

    for (scan = 0; scan < NSCAN; scan++)
    {
	/* Wait for block available */
	CHECK(wait_block_avail(hw, group, scan, verbose,
			       BLOCKSIZE, SPAN));

	/* Read the data */
	for (type = 0; type < 3; type ++)
	    for (chan = 0; chan < nchan; chan++)
	    {
		if (!active[chan])
		    continue;

		if (verbose > 1)
		    (void) printf("Reading type %d chan %d\n", type, chan);

		switch (type)
		{
		case 0:
		    data_type = E1432_TIME_DATA;
		    break;
		case 1:
		    data_type = E1432_RESAMP_DATA;
		    break;
		case 2:
		    data_type = E1432_ORDER_DATA;
		    break;
		}

		CHECK(e1432_read_float64_data(hw, chan_list[chan],
					      data_type, buf,
					      BLOCKSIZE, &trailer,
					      &count));
		if (count != BLOCKSIZE)
		{
		    (void) fprintf(stderr,
				   "%s: e1432_read_raw_data: "
				   "actual count was %ld\n",
				   progname, count);
		    return -1;
		}

		CHECK(check_trailer(&trailer, 51200, SPAN,
				    chan_list[chan],
				    data_type - E1432_TIME_DATA,
				    freq * 60));

		if (signal_list[chan] >= 0)
		{
		    /* Calculate RMS level */
		    rms = 0;
		    for (i = 0; i < BLOCKSIZE; i++)
			rms += buf[i] * buf[i];

		    if (data_type == E1432_TIME_DATA ||
			data_type == E1432_RESAMP_DATA)
			rms /= BLOCKSIZE;
		    else
			/* No divide by blocksize for freq data, but
			   scale down by two because freq data was
			   peak rather than RMS. */
			rms /= 2;

		    rms = sqrt(rms);

		    if (verbose > 2)
			(void) printf("Type %d chan %d RMS %g\n",
				      type, chan, rms);

		    /* The 0.2 limit is arbitrary */
		    if (fabs(rms - signal_list[chan]) > 0.2)
		    {
			(void) fprintf(stderr,
				       "%s: chan %d: RMS %g, expected %g\n",
				       progname, chan, rms, signal_list[chan]);
			return -1;
		    }
		}
	    }
    }

    return 0;
}

static void
usage(void)
{
    (void) fprintf(stderr,
		   "Usage: %s [-uvV] [-f freq] [-L laddr] [-n nchan]\n"
		   "\t[-N nmod]\n"
		   "\t-f: Specify tach input frequency\n"
		   "\t-L: First logical address is <laddr>, default 8\n"
		   "\t-n: Use <nchan> channels, default -1 meaning all\n"
		   "\t-N: Use <nmod> modules, default 1\n"
		   "\t-u: Print this usage message\n"
		   "\t-v: Verbose output\n"
		   "\t-V: Print version info\n",
		   progname);
    exit(2);
}

int
main(int argc, char **argv)
{
    E1432ID hw;
    SHORTSIZ16 laddr[NMOD_MAX];
    SHORTSIZ16 chan_list[NCHAN_MAX];
    double  signal_list[NCHAN_MAX];
    int     active[NCHAN_MAX];
    double  freq;
    char   *p;
    int     c, i, nmod, verbose;
    int     group, chan, nchan, offset, count;

    /* Get program name */
    progname = strrchr(argv[0], '/');
    if (progname == NULL)
	progname = argv[0];
    else
	progname++;

    /* Set option defaults */
    freq = -1;
    laddr[0] = 8;
    nchan = -1;			/* Meaning use all channels */
    nmod = 1;
    verbose = 0;
    for (chan = 0; chan < NCHAN_MAX; chan++)
	signal_list[chan] = -1;

    /* Process command-line options */
    while ((c = getopt(argc, argv, "f:L:n:N:uvVx:")) != -1)
	switch (c)
	{
	case 'f':
	    if (sscanf(optarg, "%lg", &freq) != 1)
	    {
		(void) fprintf(stderr,
			       "%s: invalid frequency: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'L':
	    laddr[0] = (SHORTSIZ16) strtol(optarg, &p, 0);
	    if (optarg == p || laddr[0] < 0 || laddr[0] > 255)
	    {
		(void) fprintf(stderr,
			       "%s: invalid logical address: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'n':
	    nchan = strtol(optarg, &p, 0);
	    if (optarg == p || nchan < -1 || nchan > NCHAN_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of channels: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'N':
	    nmod = strtol(optarg, &p, 0);
	    if (optarg == p || nmod < 0 || nmod > NMOD_MAX)
	    {
		(void) fprintf(stderr,
			       "%s: invalid number of modules: '%s'\n",
			       progname, optarg);
		usage();
	    }
	    break;
	case 'v':
	    verbose++;
	    break;
	case 'V':
	    (void) printf("%s\n", rcsid);
	    exit(EXIT_SUCCESS);
	case 'x':
	    offset = 0;
	    for (chan = 0; chan < NCHAN_MAX; chan++)
	    {
		if (sscanf(optarg + offset, "%lg%n",
			   &signal_list[chan], &count) != 1)
		{
		    if (strlen(optarg + offset) > 0)
		    {
			(void) fprintf(stderr,
				       "%s: invalid expected signal level: '%s'\n",
				       progname, optarg);
			usage();
		    }
		    break;
		}
		offset += count;
		count = 0;
		(void) sscanf(optarg + offset, " ,%n", &count);
		offset += count;
	    }
	    break;
	case 'u':
	default:
	    usage();
	}

    if (argc > optind)
    {
	(void) fprintf(stderr, "%s: extra command-line arguments\n",
		       progname);
	usage();
    }

    /* Assume logical addresses are consecutive */
    for (i = 1; i < nmod; i++)
	laddr[i] = laddr[i - 1] + 1;

    /* Run the measurement */
    if (init(nmod, laddr, &hw, &group, &nchan, chan_list) < 0)
	return EXIT_FAILURE;
    if (setup(hw, group, nchan, chan_list) < 0)
	return EXIT_FAILURE;

    if (set_active1(hw, nchan, chan_list, active) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list,
	    signal_list, active, freq, verbose) < 0)
	return EXIT_FAILURE;

    if (set_active2(hw, nchan, chan_list, active) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list,
	    signal_list, active, freq, verbose) < 0)
	return EXIT_FAILURE;

    if (set_active3(hw, nchan, chan_list, active) < 0)
	return EXIT_FAILURE;
    if (run(hw, group, nchan, chan_list,
	    signal_list, active, freq, verbose) < 0)
	return EXIT_FAILURE;

    return EXIT_SUCCESS;
}
